(** This modules implements shared forests and shared forest contexts
    (zippers for shared forests). *)

module Log : UtilsLib.Xlog.MyLOG
(** [Log] is the log module for SharedForest *)
               
module Make (W : Weight.Weight_sig) : 
sig
  type address = (int * int) list
  (** This type is the type of addresses of forests. It is a list of
      (position in the forest,position as a child). *)

  type relative_path = int * address
  (** This is the type of relative path from one forest to another
      one. The first argument is the number of steps to move up, then
      second argument is the address to reach from this point. *)

  val diff : address -> address -> relative_path
  (** [diff add add'] returns the relative path to go from the forest
     (subtree) wich occurs at address [add] to the forest (subtree)
     wich occurs at address [add']. *)

  val pp_path : Format.formatter -> relative_path -> unit
  (** [pp_path fmt p] pretty prints the path [p] on the formatter
      [fmt]. *)
    
  val pp_address : Format.formatter -> address -> unit
  (** [pp_path fmt add] pretty prints the address [add] on the formatter
      [fmt]. *)

  type weight = W.w
  (** This type is the type for weights of trees. It is used to
     enumerate trees in the resumptions in the order specified by [W]
     *)

  val pp_weight : Format.formatter -> weight -> unit
  (** [pp_path fmt w] pretty prints the weight [w] on the formatter
     [fmt]. *)

  type 'a forest = 'a forest_tree list
  and 'a forest_tree = Node of ('a * 'a child list)
  and 'a child =
    | Forest of 'a forest
    | Link_to of relative_path
    (** The type of the forests, made of the type of their nodes and
        the type of their children *)
        
  type 'a childList_context =
    { siblings: 'a child ListContext.focused_list;
      parent_label : 'a;
      focus: 'a child;}
  (** The type of the context for child lists (cannot be
      straightforwadly list context because elements can be either
      forests or links to other places in the shared forest *)
    
  
  type 'a forest_context_info =
    { label : 'a;
      (* label is the label of the node from which the context was
         made *)
      children : 'a child ListContext.focused_list;
      (* the focused list of children of the tree. Just as for tree
         zippers *)
      alternatives: 'a forest_tree ListContext.focused_list;
      (* the focused list of the focused children: a forest *)
      alt_num : int;
      (* the position of the tree under focus in the current forest *)
      context : 'a forest_context;
      (* the forest context *)
      suspended_computation :  (('a forest_context * 'a forest_tree) * 'a childList_context) option;
      (* a local context describing the state from which the current
         focused forest was reach after following a [Link_to]
         specification. Because when moving up (or right) from such
         node, we should take into account the forest tree whose the
         [Link_to] was a child (and its context), not the current one.
      *)
      address : address ;
      (* The (absolute) address of the current hole for forest_tree
         (**** IN THE REVERSE ORDER ****)*)
    }
  and 'a forest_context =
    | Top of ('a forest_tree ListContext.focused_list * int * (('a forest_context * 'a forest_tree) * 'a childList_context) option)
    (* invariant: the second element should be the position of the
       current focused tree of the forest) *)
    | Zip of 'a forest_context_info
    (** The type for shared forest contexts. Using a record to better
       distinguish the different components *)

  module Resumptions : Resumptions.Resumptions_sig
  (** A resumption store module for the weighting scheme being used *)

  val init : alt_max:int -> (Format.formatter -> 'a -> unit) -> 'a forest_tree list -> 'a Resumptions.resumptions
  (** [init ~alt_max ppf forest] builds the resumptions with a focused
     forest focusing on each of the tree of [forest]. [10^alt_max] is
     the max number of delayed computations before moving to
     non-regular sorting. [ppf] is a pretty printer for ['a] type
     values.*)

  val resume :
    'a Resumptions.resumptions -> ('a TreeContext.Tree.tree * W.w) option * 'a Resumptions.resumptions
  (** [resume resumptions] returns a pair [(Some (t,n),resumptions')]
     where [t] of size [n] is extracted from [resumptions], the latter
     being updated with possible alternatives met in building [t] to
     produce [resumptions']. It returns [(None,[])] if no tree can be
     extracted *)

  val pp_trees : (Format.formatter -> 'a -> unit) -> Format.formatter -> 'a Resumptions.resumptions -> unit
  (** [pp_tree pp fmt r] pretty prints all the trees by generated by
      the resumptions on the formatter [fmt] using the pretty printer
      for tree nodes [pp] *)
    
  val pp_forest : (Format.formatter -> 'a -> unit) -> Format.formatter -> 'a forest -> unit
  (** [pp_tree pp fmt f] pretty prints the forest [f] (with actual
      link-to node) on the formatter [fmt] using the pretty printer for
      tree nodes [pp] *)
end


module SharedForest : module type of Make (Weight.Weight_as_Depth_and_Size)
