(** This module implements the [expand] algorithm of {{:
    http://www.collegepublications.co.uk/downloads/ifcolog00013.pdf#page305}Makoto
    Kanazawa's article "Parsing and generation as Datalog query
    evaluation" } to deal with almost linear grammars.
*)

open Lambda

module Log : UtilsLib.Xlog.MyLOG
(** [Log] is the log module for Expand *)

(** The {!Address} module implements addresses as used in Makoto's
    paper *)
module Address :
sig
  type t
  (** [t] is the type of addresses *)

  val make_add : int list -> t
  (** [make_add lst] turns a list of [0]s and [1]s into an address.
      Raises [Invalid_argument] if [lst] contains a vaue different
      from [0] or [1].*)

  val pp : Format.formatter -> t -> unit
  (** [pp fmt add] pretty prints the address [add] in [fmt] *)
end

(** The {!AddSet} module implements a trie data structure to represent
    sets of {!Address.t} addresses *)
module AddSet :
sig
  type t
  (** [t] is the type of sets of addresses *)

  val empty : t
  (** [empty] is the empty set of addresses *)

  val mem : Address.t -> t -> bool
  (** [mem add s] returns [true] if [add] is in the set [s] and
      [false] otherwise. *)

  val add : Address.t -> t -> t
  (** [add a s] adds the address [a] into the set [s]. [s] is
      unchanged if [a] already belonged to [s].*)

  val iter : (Address.t -> unit) -> t -> unit
  (** [iter f s] iterates the function [f] on every elements of [s].*)

end

val collapse : ?consts:Lambda.consts -> Lambda.term -> Lambda.term option
(** [collapse ~consts t] returns [None] if [t] is unchanged through
    the collapse algorithm (i.e., no subterm of atomic type occurs at
    least twice in [t]), and [Some u] where [u] is the results of the
    (recursive) collapse algorithm.

    {b It is expected that [t] does not contain unexpanded defined
    constants.}

    If [consts] is provided, the mapping from constant ids to strings
    (in some signature) is used to pretty prints terms if {!Log} log
    level is set to some adequate level. Otherwise, each constant is
    printed as [Const[i]].

 *)
