(**************************************************************************)
(*                                                                        *)
(*                 ACG development toolkit                                *)
(*                                                                        *)
(*                  Copyright 2008-2024 INRIA                             *)
(*                                                                        *)
(*  More information on "https://acg.loria.fr/"                           *)
(*  License: CeCILL, see the LICENSE file or "http://www.cecill.info"     *)
(*  Authors: see the AUTHORS file                                         *)
(*                                                                        *)
(*                                                                        *)
(*                                                                        *)
(*                                                                        *)
(*                                                                        *)
(**************************************************************************)

(** This module implements basic operations on lambda terms. It uses
    de Bruijn indices.*)

open Abstract_syntax

(** This is the actual module *)
module Lambda : sig

  (** This signature represents maps from the de Bruijn indices of a
      variable to some info that was recorded when the corresponding
      binder was crossed. *)
  module type VarEnvSig =
    sig
      type level = int
      type info
      type t 
      val empty : t
      val add : info -> t -> t
      val set : level -> info -> t -> t
      val get : level -> t -> info
      val current_level : t -> level
      val get_opt : level -> t -> info option
      val exists : (info -> bool) -> t -> bool
      val shift : info:info -> level:level -> t -> t
    (** [shift ~info ~level e] changes [e] into [e'] where: if level
        [l < level] maps to info [i], then [l] maps to [i] in [e'] as
        well. If [l >= level] maps to info [i], then [l+1] maps to
        [i]. Finally, in [e'], [level] maps to [info].*)

      val pp : Format.formatter -> t -> unit
    end

  (** This modules is a functor that returns a map from the de Bruijn
      indices of a variable to some info that was recorded when the
      corresponding binder was crossed. *)
  module MakeVarEnv (I: sig
                   type info
                   val pp : Format.formatter -> info -> unit
                 end) : VarEnvSig 
         with type info = I.info

  (** This module implements a map from the de Bruijn indices of a
      variable to strings that was recorded when the corresponding
      binder was crossed.*)
  module VNEnv : VarEnvSig with 
           type info = string type env = VNEnv.t

  type kind = Type | Depend of stype * kind
  (* the kind of a dependant type *)
  
  and stype =
    | Atom of int (* atomic type *)
    | DAtom of int (* defined atomic type *)
    | LFun of stype * stype (* linear functional type *)
    | Fun of stype * stype (* non linear functional type *)
    | Dprod of string * stype * stype (* dependant product *)
    | Record of int * stype list (* records *)
    | Variant of int * stype list (* variants *)
    | TAbs of string * stype (* type abstraction *)
    | TApp of stype * term
  (* type application *)

  and term =
    | Var of int (* lambda variable *)
    | LVar of int (* linear lambda variable *)
    | Const of int (* constant *)
    | DConst of int (* defined constant *)
    | Abs of string * term (* lambda-abstraction *)
    | LAbs of string * term (* linear lambda abstraction *)
    | App of term * term (* application *)
    | Rcons of int * term list (* record constructor:         *)
    (* - the integer is the tag of *)
    (*   the corresponding type.   *)
    | Proj of int * int * term (* projection:                        *)
    (* - the first integer is the tag of  *)
    (*   the corresponding type;          *)
    (* - the second integer is the index  *)
    (*   of the projection.               *)
    | Vcons of int * int * term (* variant constructor:               *)
    (* - the first integer is the tag of  *)
    (*   the corresponding type;          *)
    (* - the second integer is the number *)
    (*   of the constructor.              *)
    | Case of int * term * (string * term) list
    (* case analysis:              *)
    (* - the integer is the tag of *)
    (*   the corresponding type.   *)
    | Unknown of int
  (* meta-variable - used in higher-order  *)
  (* matching                              *)

  type consts = int -> Abstract_syntax.syntactic_behavior * string

  (*  val env_to_string : env -> string *)
  val generate_var_name : string -> env * env -> string
  val unfold_labs : string list -> env * env -> term -> string list * (env * env) * term
  val unfold_abs : string list -> env * env -> term -> string list * (env * env) * term
  val unfold_app : term list -> term -> term list * term
  val is_binder : int -> consts -> bool
  val is_infix : int -> consts -> bool
  val is_prefix : int -> consts -> bool

  val unfold_binder :
    int ->
    consts ->
    string list ->
    env * env ->
    term ->
    string list * (env * env) * term

  val pp_kind : consts -> Format.formatter -> kind -> unit
  val pp_type : consts -> Format.formatter -> stype -> unit
  val pp_term : ?env:(env*env) -> consts -> Format.formatter -> term -> unit
  val raw_to_string : term -> string
  val raw_type_to_string : stype -> string
  val raw_to_caml : term -> string
  val raw_type_to_caml : stype -> string
  val normalize : ?id_to_term:(int -> term) -> term -> term
  val unlinearize_term : term -> term
  val unlinearize_type : stype -> stype

  val eta_long_form : term -> stype -> (int -> stype) -> term
  (** [eta_long_form t ty type_of_cst] returns the eta-long form of
      [t] with respect of type [ty]. [t] is supposed to be in
      beta-normal form and all the definitions of [t] and [ty] should
      have been unfolded. [type_of_cst i] returns
      the type (with unfolded definitions) of the constant whose id is
      [i]. [i] is supposed to be an actual id of a constant.*)

  val is_2nd_order : stype -> (int -> stype) -> bool
  (** [is_2nd_order ty type_definition] returns [true] if [ty] is 2nd
      order. [ty] should have been unfolded and [type_definition i] is
      returns the unfolded type of a defined type ([DAtom]) whose id
      is [i]. [i] is supposed to be an actual id of such a defined type.*)

  val is_atomic : stype -> (int -> stype) -> bool
  (** [is_atomic ty type_definition] returns [true] if [ty] is
     atomic. [type_definition i] returns the unfolded type of a
     defined type ([DAtom]) whose id is [i]. [i] is supposed to be an
     actual id of such a defined type.*)

  val size : id_to_term:(int -> term) -> term -> int * int

  val alpha_eq : term -> term -> bool
  (** [alpha_eq t u] returns [true] if [t] and [u] are
      alpha-equivalent, [false] otherwise. *)

  val equal :
    id_to_term:(int -> term) ->
    type_of_const:(int -> stype) -> (term * stype) -> (term * stype) -> bool
end
