(** This module provides pretty printing formatters and utilities *)

val pp_list :
  ?sep:(unit, Format.formatter, unit, unit, unit, unit) format6 ->
  ?terminal:(unit, Format.formatter, unit, unit, unit, unit) format6 ->
  (Format.formatter -> 'a -> unit) ->
  Format.formatter ->
  'a list ->
  unit
(** [pp_list ~sep ppf fmt l] pretty prints on the formater [fmt] the
   items of the list [l], each of them being pretty-printed by
   [ppf]. Example usage: [pp_list ~sep:",@\n" Format.pp_print_int
   Format.std_formatter [1;2;3] ] *)

val pp_list_i :
  ?sep:(unit, Format.formatter, unit, unit, unit, unit) format6 ->
  ?terminal:(unit, Format.formatter, unit, unit, unit, unit) format6 ->
  (Format.formatter -> (int * 'a) -> unit) ->
  Format.formatter ->
  'a list ->
  unit
(** [pp_list_i ~sep ppf fmt l] pretty prints on the formater [fmt] the
   items of the list [l], each of them, together with its position
   (starting at 1) being pretty-printed by [ppf].*)

val pp_text : Format.formatter -> string -> unit
(** [pp_text fmt t] pretty prints on the formater [fmt] the text of of
    the string [t] where each space in [t] is made a space break. *)

val resize_formatter :
  ?width:int -> ?formatter:Format.formatter -> unit -> unit
(** [resize_formatter ~width ~formatter ()] sets the max output width of the
    formatter [formatter] to [width]. Argument [formatter] defaults to
    [Format.std_formatter]. Argument [width] defaults to the current terminal
    width if [stdout] is a tty, or [Format.pp_infinity - 1] otherwise. *)

val no_pp : ?formatter:Format.formatter -> unit -> unit
(** [no_pp ()] sets the max output width of the formatter [formatter] to
    [Format.pp_infinity - 1] so that break hints are basically not used,
    and no pretty printing information is used but semantic tags.
    Argument [formatter] defaults to [Format.std_formatter]. *)

val format_of_list :
  Format.formatter -> string -> ('a -> string) -> 'a list -> unit

val app_pp : Format.formatter -> string -> unit
(** [app_pp ppf s] pretty prints the string [s] on the formatter [ppf]
    surrounded by the application semantic tag <app> *)

val err_pp : Format.formatter -> string -> unit
(** [err_pp ppf s] pretty prints the string [s] on the formatter [ppf]
    surrounded by the error semantic tag <err> *)

val warn_pp : Format.formatter -> string -> unit
(** [warn_pp ppf s] pretty prints the string [s] on the formatter [ppf]
    surrounded by the warning semantic tag <warn> *)

val info_pp : Format.formatter -> string -> unit
(** [info_pp ppf s] pretty prints the string [s] on the formatter [ppf]
    surrounded by the information semantic tag <info> *)

val debug_pp : Format.formatter -> string -> unit
(** [debug_pp ppf s] pretty prints the string [s] on the formatter [ppf]
    surrounded by the debug semantic tag <debug> *)

val fun_pp : Format.formatter -> string -> unit
(** [fun_pp ppf s] pretty prints the string [s] on the formatter [ppf]
    surrounded by the semantic tag <fun> *)

val sig_pp : Format.formatter -> string -> unit
(** [sig_pp ppf s] pretty prints the string [s] on the formatter [ppf]
    surrounded by the semantic tag <sig> *)

val lex_pp : Format.formatter -> string -> unit
(** [lex_pp ppf s] pretty prints the string [s] on the formatter [ppf]
    surrounded by the semantic tag <lex> *)

val term_pp : Format.formatter -> string -> unit
(** [terms_pp ppf s] pretty prints the string [s] on the formatter [ppf]
    surrounded by the semantic tag <terms> *)

val arg_pp : Format.formatter -> string -> unit
(** [arg_pp ppf s] pretty prints the string [s] on the formatter [ppf]
    surrounded by the semantic tag <arg> *)

val binary_pp : Format.formatter -> string -> unit
(** [bin_pp ppf s] pretty prints the string [s] on the formatter [ppf]
    surrounded by the semantic tag <bin> *)
