open Js_of_ocaml
open Scripting

let dirs = [ "" ]
let step_by_step = true
let with_magic = false
let rendering_config = SvgLib.Svg.get_config ~filename:None ~dirs
let width = None
let config = Scripting.Config.{ dirs; rendering_config; step_by_step ; with_magic; width }
let env = ref
  Scripting.Environment.
    {
      config;
      acg_env = AcgData.Environment.Environment.empty;
      functions = Scripting.Functions.def_fun_list (Some config);
      last_value = None;
    }

let init () =
  Scripting.AcgLog.set_acg_log ();
  Random.self_init ()

let parse command =
  let lexbuf = Sedlexing.Utf8.from_string command in
  let parser =
    MenhirLib.Convert.Simplified.traditional2revised CodeParser.interactive_command in

  try
    parser
      (fun () -> match Lexer.lex lexbuf with
      | Lexer.Token (tok, _str) ->
          let pstart, pend = Sedlexing.lexing_positions lexbuf in
          tok, pstart, pend
      | Lexer.PartialToken (_, err) ->
          let loc = Sedlexing.lexing_positions lexbuf in
          Errors.LexingErrors.emit err ~loc)
  with
    | CodeParser.Error s ->
      let loc = Sedlexing.lexing_positions lexbuf in
        Errors.(SyntaxErrors.emit (Syntax_l.SyntaxError s) ~loc)

let setup_logger ?width f =
  UtilsLib.PPUtils.resize_formatter ~formatter:f ?width ();
  UtilsLib.Xlog.setup_log
    ~app:"acg"
    ~render_mark:HTML_Rendering.render_mark
    ~colored:true
    Logs.Warning
    f

let exec interactive input max_cols =
  (try
    let tmp_buffer = Buffer.create 100 in
    let tmp_formatter = Format.formatter_of_buffer tmp_buffer in
    setup_logger ~width:max_cols tmp_formatter;
    let command = parse input in
    let (res_val_list, res_env) = command !env in
    env := { res_env with Environment.last_value =
      if res_val_list = None then res_env.Environment.last_value else res_val_list };
    let text = Buffer.contents tmp_buffer in
    setup_logger ~width:max_cols (Postmessage_formatter.f (fun m -> Messages.OutputText m));
    if interactive then
      Logs.app (fun m -> m "%s%s\n%s" UtilsLib.Error.base_prompt input text)
    else
      Logs.app (fun m -> m "%s" text);
    Option.iter (fun v -> Scripting.Value.print true v) res_val_list
  with
    | e ->
      setup_logger ~width:max_cols (Postmessage_formatter.f (fun m -> Messages.OutputText m));
      if interactive then UtilsLib.Error.print_error e (Some input));
  Worker.post_message Messages.ExecCommandDone

let execute_command input max_cols =
  exec true input max_cols

let get_env max_cols =
  setup_logger ~width:max_cols (Postmessage_formatter.f (fun m -> Messages.OutputEnvText m));
  Scripting.Environment.short_print !env

let clear_env () =
  env := { !env with Scripting.Environment.acg_env = AcgData.Environment.Environment.empty }

let compile_grammar grammar =
  setup_logger Null_formatter.f;
  let result =
    try
      let new_env = snd (Dump.load_env_str ~with_magic:false grammar !env.Scripting.Environment.acg_env) in
      env := { !env with Scripting.Environment.acg_env = new_env };
      Result.Ok ()
    with
    | UtilsLib.Error.AcgtkError (kind, pps, l) ->
      let message = Format.asprintf "%s error: %t" kind pps in
      Result.Error (message, l)
    | _ -> Result.Error ("Unknown error", None) in
  Worker.post_message (Messages.CompileGrammarDone result)
