open Js_of_ocaml
open Js_of_ocaml_tyxml.Tyxml_js

let environment_update = ref (fun () -> ())

let info =
  Html.(div [ txt "✔" ])
let info_dom = To_dom.of_div info

let compile_button =
  Html.(button [ txt "Compile" ])
let compile_button_dom = To_dom.of_button compile_button

let textarea =
  Html.(div ~a:[ a_class [ "textarea" ; "fill" ; "pad_6" ] ; a_contenteditable true ] [ div [] ])
let textarea_dom = To_dom.of_div textarea

let load_input =
  Html.(input ~a:[ a_input_type `File ; a_style "display: none" ] ())
let load_input_dom = To_dom.of_input load_input

let load_button =
  Html.(button ~a:[ a_onclick (fun _ -> load_input_dom##click; true) ] [ txt "Load from file" ])
let load_button_dom = To_dom.of_button load_button

let save_button =
  Html.(button [ txt "Save to file" ])
let save_button_dom = To_dom.of_button save_button

let file_name = ref "grammar.acg"

type grammar_state =
  | Modified
  | UpToDate
  | Compiled
  | Error

let update_callback state =
  match state with
  | State.Idle ->
    compile_button_dom##.disabled := Js._false;
    load_button_dom##.disabled := Js._false
  | State.Compiling
  | State.Executing ->
    compile_button_dom##.disabled := Js._true;
    load_button_dom##.disabled := Js._true

let update_dom state =
  match state with
  | Modified -> info_dom##.innerHTML := Js.string "✱"
  | UpToDate -> info_dom##.innerText := Js.string ""
  | Compiled -> info_dom##.innerText := Js.string "✔"
  | Error -> info_dom##.innerText := Js.string "✘"

let state = ref Compiled
let set_state new_state = state := new_state; update_dom new_state

let compile_callback = ref (fun () -> ())

let get () =
  let clean_list = Js_utils.clean_node_list textarea_dom in
  Js_utils.replace_children textarea_dom clean_list;
  Js_utils.node_to_text textarea_dom

let compile callback =
  match !state with
  | Modified ->
    let grammar = get () in
    if String.trim grammar <> "" then
      (set_state UpToDate;
      State.set_state State.Compiling;
      compile_callback := callback;
      Interface.compile_grammar grammar)
    else
      (set_state Compiled;
      callback ())
  | Compiled
  | UpToDate ->
    callback ()
  | Error -> ()

let force_compile callback =
  state := Modified;
  compile callback

let load () =
  file_name :=
    Js.to_string (Js.Opt.get (load_input_dom##.files##item 0) (fun () -> assert false))##.name;
  Js_utils.read_fileinput load_input_dom
    (fun text ->
      Js_utils.replace_children textarea_dom (Js_utils.text_to_nodes text);
      force_compile (fun () -> ()))

let save () =
  let uri = Js_utils.generate_blob (Js.string (get ())) in
  Js_utils.download_file uri !file_name

let compile_callback result =
  State.set_state State.Idle;
  match result with
  | Result.Ok () ->
    !environment_update ();
    if !state = UpToDate then set_state Compiled;
    !compile_callback ()
  | Result.Error (error_message, Some loc) ->
    if !state = UpToDate then set_state Error;
    Js_utils.highlight_error textarea_dom error_message loc
  | Result.Error (_error_message, None) ->
    if !state = UpToDate then set_state Error

let init environment_update_callback =
  compile_button_dom##.onclick := Dom_html.handler (fun _ -> compile (fun () -> ()); Js._true);
  (Js.Unsafe.coerce textarea_dom)##.oninput := Dom_html.handler (fun _ -> set_state Modified; Js._true);
  load_input_dom##.onchange := Dom_html.handler (fun _ -> load (); Js._true);
  save_button_dom##.onclick := Dom_html.handler (fun _ -> save (); Js._true);
  environment_update := environment_update_callback
