(** This module implements the grammar text box. *)

val init : (unit -> unit) -> unit
(** [init environment_update_callback] should be called before the page is set
  up. [environment_update_callback] is a function that will be called when the
  grammar is successfully compiled. Its purpose is to update the environment
  textarea (because the ACG environment may have changed). *)

val textarea :
  [> Html_types.div ] Js_of_ocaml_tyxml.Tyxml_js.Html.elt
(** [textarea] is an HTML div that can be modified by the user to write
  the grammar. *)

val info :
  [> Html_types.div ] Js_of_ocaml_tyxml.Tyxml_js.Html.elt
(** [info] is an HTML div which displays the current status of the
  grammar (modified, compiling, compiled, error). *)

val compile_button :
  [> Html_types.button ] Js_of_ocaml_tyxml.Tyxml_js.Html.elt
(** [compile_button] is an HTML button that starts the compilation of the
  current grammar. *)

val load_button :
  [> Html_types.button ] Js_of_ocaml_tyxml.Tyxml_js.Html.elt
(** [load_button] is an HTML button that prompts the user to select a file. The
  file will be loaded in [textarea]. *)

val load_input :
  [> Html_types.input ] Js_of_ocaml_tyxml.Tyxml_js.Html.elt
(** [load_input] is an HTML input that displays nothing on the page but must be
  present for [load_button] to work. *)

val save_button :
  [> Html_types.button ] Js_of_ocaml_tyxml.Tyxml_js.Html.elt
(** [save_button] is an HTML button that saves the current content of [textarea]
  to a file. *)

val compile : (unit -> unit) -> unit
(** [compile callback] compiles the current grammar (in [textarea]) if it has
  changed since last compilation. The function [callback] will be called if the
  compilation is successful. *)

val force_compile : (unit -> unit) -> unit
(** [force_compile callback] compiles the current grammar (in [textarea]) even
  if it has not been modified since last compilation. It otherwise behaves
  identically to [compile]. *)

val compile_callback :
  ( unit,
    string * (Lexing.position * Lexing.position) option )
  Result.t ->
  unit

val update_callback : State.state -> unit
(** [update_callback state] changes the availability of the buttons depending
  on the state [state]. It should only be called by [State.set_state] when the
  state changes. *)
