(** This module sets up and communicates with the Web Worker that performs the
  ACG computations. This allows to have long computations that do not freeze the
  page. This module is just the interface, the actual code that run in the Web
  Worker is in the {: ../acg} directory. *)

val init :
  ((unit, string * (Lexing.position * Lexing.position) option) result -> unit) ->
  (string -> unit) ->
  (string -> unit) -> (unit -> unit) -> (unit -> unit) -> unit
(** [init compile_callback answer_callback environment_callback execute_callback restart_callback]
  starts the Web Worker. It will store and call the provided callback functions
  when their corresponding task is done. *)

val restart : unit -> unit
(** [restart ()] restarts the Web Worker. This may be used to interrupt a
  computation that is too long or infinite. *)

val set_max_output_cols : int -> unit
(** [set_max_output_cols n] configures ACGtk output to a maximum width of [n]. *)

(** The following functions are the functions of the interface that requests the
  Web Worker to perform the actual computations. This means that these requests
  do not freeze the page, even if they take a long time. When the request is
  done, the corresponding callback (that has been set up with the [init]
  function) will be called. *)

val compile_grammar : string -> unit
(** [compile_grammar grammar] starts the compilation of the grammar [grammar]. *)

val exec_command : string -> unit
(** [exec_command command] starts the computation of the command [command]. *)

val clear_env : unit -> unit
(** [clear_env ()] clears the ACG environment. *)

val get_env : unit -> unit
(** [gen_env ()] prints the current ACG environment. *)
