open Js_of_ocaml
open Js_of_ocaml_tyxml.Tyxml_js

let get_embedded_script filter =
  let scriptList = Dom_html.document##querySelectorAll (Js.string filter) in
  Js.Opt.get (Js.Opt.get (scriptList##item 0) (fun _ -> assert false))##.textContent (fun _ -> assert false)

let generate_blob text =
  let blob = Js.Unsafe.new_obj (Js.Unsafe.global##._Blob) [| Js.Unsafe.inject (Js.array [| text |]) |] in
  (Js.Unsafe.coerce Dom_html.window)##.URL##createObjectURL blob

let read_fileinput fileinput callback =
  let fr = Js.Unsafe.new_obj (Js.Unsafe.global##._FileReader) [| |] in
  let file = Js.Opt.get (fileinput##.files##item 0) (fun _ -> assert false) in
  let _ = Js.Unsafe.meth_call fr "readAsText" [| Js.Unsafe.coerce file |] in
  let () = fr##.onload := (fun _ -> callback fr##.result) in ()

let download_file uri file_name =
  let link = Html.(a ~a:[ a_href uri ; a_download (Some file_name) ] []) in
  let link_dom = To_dom.of_a link in
  link_dom##click

let get_event_key e = Js.Optdef.get (e##.key) (fun _ -> Js.string "")

let line_to_html l =
  if l = "" then Html.(div [br ()])
  else
    let div = Dom_html.createDiv Dom_html.window##.document in
    let () = div##.innerHTML := (Js.string l) in
    Of_dom.of_div div

let text_to_nodes text =
  let text = if String.ends_with ~suffix:"\n" text then String.sub text 0 ((String.length text) - 1) else text in
  let s = String.split_on_char '\n' text in
    (List.map (fun l -> To_dom.of_node (line_to_html l)) s)

let text_to_node text =
  let text = if String.ends_with ~suffix:"\n" text then String.sub text 0 ((String.length text) - 1) else text in
  let s = String.split_on_char '\n' text in
    To_dom.of_node (Html.div (List.map line_to_html s))

let rec clean_node_list_aux toplevel node =
  let children = Dom.list_of_nodeList (node##.childNodes) in
  List.fold_left
    (fun acc n ->
      match Js.Opt.to_option (Dom.CoerceTo.text n) with
      | Some n_text -> acc @ [ To_dom.of_node Html.(div [ txt (Js.to_string n_text##.data) ]) ]
      | None ->
        match Js.Opt.to_option (Js.Opt.bind (Dom.CoerceTo.element n) (fun e -> Dom_html.CoerceTo.div (Dom_html.element e))) with
        | Some n_div ->
          if Js.to_string n_div##.id = "error_msg"
            then acc
            else acc @ (clean_node_list_aux false n_div)
        | None ->
          match Js.Opt.to_option (Js.Opt.bind (Dom.CoerceTo.element n) (fun e -> Dom_html.CoerceTo.br (Dom_html.element e))) with
          | Some _ -> if acc = [] || toplevel then acc @ [ To_dom.of_node Html.(div [ br () ]) ] else acc
          | None -> acc)
    [] children

let clean_node_list node =
  clean_node_list_aux true node

let node_to_text node =
  let children = Dom.list_of_nodeList (node##.childNodes) in
  List.fold_left
    (fun acc n ->
      match Js.Opt.to_option (Js.Opt.bind (Dom.CoerceTo.element n) (fun e -> Dom_html.CoerceTo.div (Dom_html.element e))) with
      | Some n_div -> acc ^ (Js.to_string (Js.Opt.get n_div##.textContent (fun _ -> Js.string ""))) ^ "\n"
      | None -> acc)
    "" children

let generate_error message =
  Html.(div ~a:[ a_id "error_msg" ; a_class ["floating"] ; a_contenteditable false ] [div [txt message]])

let highlight_node node =
  match Js.Opt.to_option (Js.Opt.map (Dom.CoerceTo.element node) (fun e -> Dom_html.element e)) with
  | Some n_elt ->
    n_elt##.classList##add (Js.string "lightred");
    n_elt##.classList##add (Js.string "relative")
  | None -> ()

let attach_error node error_message =
  match Js.Opt.to_option (Js.Opt.map (Dom.CoerceTo.element node) (fun e -> Dom_html.element e)) with
  | Some n_elt ->
    ignore (n_elt##appendChild (To_dom.of_node (generate_error error_message)))
  | None -> ()

let highlight_error (node : Dom_html.divElement Js.t) error_message (l1, l2) =
  let ls = l1.Lexing.pos_lnum - 1 in
  let le = l2.Lexing.pos_lnum in
  List.iter
    (fun i ->
      let error_node =
        match Js.Opt.to_option (node##.childNodes##item i) with
        | Some node -> node
        | None -> Js.Opt.get node##.lastChild (fun _ -> assert false) in
      highlight_node error_node;
      if i = ls then attach_error error_node error_message)
    (List.init (le - ls) (fun i -> ls + i))

let replace_children node nodes =
  let () = List.iter (fun c -> ignore (node##removeChild c)) (Dom.list_of_nodeList node##.childNodes) in
  let () = List.iter (fun c -> ignore (node##appendChild c)) nodes in
  ()

let log text = Console.console##log (Js.string text)

let is_worker () =
  try let _ = Dom_html.window##.name in false with _ -> true
